package latexDraw.ui.components.progressbars;

import java.awt.event.ActionEvent;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JProgressBar;

import latexDraw.figures.Figure;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.parsers.Parser;
import latexDraw.parsers.pstricks.LaTeXDrawPSTricksParserActions;
import latexDraw.parsers.pstricks.PSTricksParameters;
import latexDraw.parsers.pstricks.PSTricksParser;
import latexDraw.ui.CodePanel;
import latexDraw.ui.DrawContainer;
import latexDraw.ui.DrawPanel;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.util.LaTeXDrawResources;

/** 
 * Defines a progress bar manager of PSTricks import.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 * <br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 * <br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 * <br>
 * 02/04/08<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class PSTProgressBarManager extends AbstractProgressBarManager
{
	/** The TeX parser */
	private transient PSTricksParser texParser;
	
	/** The code to parse. */
	protected String codeToParse;
	
	
	public PSTProgressBarManager(LaTeXDrawFrame frame)
	{
		super(frame);
		
		codeToParse = null;
	}

	
	@Override
	public void actionPerformed(ActionEvent e)
	{
		super.actionPerformed(e);
		
		Object src = e.getSource();
		
		if(src instanceof JButton && LaTeXDrawResources.LABEL_STOP.equals(((JButton)(src)).getActionCommand()))
			if(parsingThread!=null) texParser.stop();
	}



	public void run()
	{
		if(codeToParse==null && fileToParse==null)
			return ;
		
		JProgressBar progressBar 	= frame.getProgressBar();
		JButton stopButton 			= frame.getStopButton();
		LaTeXDrawPSTricksParserActions ltpa = new LaTeXDrawPSTricksParserActions();
		texParser = new PSTricksParser(ltpa);
		progressBar.setValue(0);
		stopButton.setVisible(true);
		progressBar.setVisible(true);
		
		int res;
		
		if(fileToParse!=null)
			 res = texParser.parse(fileToParse, frame, progressBar);
		else res = texParser.parse(codeToParse, frame, progressBar);
			
		Vector<Figure> fig = ltpa.getFigures();
		
		switch(res)
		{
			case Parser.ERROR_PARSING :
				
				frame.setStatusBarText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.34")); //$NON-NLS-1$
				break;
					
			case Parser.CANCEL_PARSING :
				
				frame.setStatusBarText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.35")); //$NON-NLS-1$
				break;
					
			case Parser.OK_PARSING :
				
				if(fig==null || fig.isEmpty())
				{
					JOptionPane.showMessageDialog(frame, LaTeXDrawLang.getString1_6("LaTeXDrawFrame.30") //$NON-NLS-1$
					+ LaTeXDrawResources.MAIL_ME + LaTeXDrawLang.getString1_6("LaTeXDrawFrame.31"), //$NON-NLS-1$
					LaTeXDrawLang.getString1_6("LaTeXDrawFrame.32"),   //$NON-NLS-1$ 
					JOptionPane.WARNING_MESSAGE );
					frame.setStatusBarText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.33")); //$NON-NLS-1$
				}
				else
				{
					LaTeXDrawPSTricksParserActions.optimizeCode(fig);
					DrawPanel drawPanel = frame.getDrawPanel();
					CodePanel codePanel = frame.getCodePanel();
					DrawContainer draw  = drawPanel.getDraw();
					
					for(Figure f : fig)
						draw.addFigure(f, frame.getUndoManager());
					
					draw.recenterFigures();
					codePanel.setPositionHoriToken(PSTricksParameters.isCentered ? 
										CodePanel.TOKEN_HORI_CENTER_START : CodePanel.TOKEN_HORI_VERT_DEFAULT);
					codePanel.setPositionVertToken(PSTricksParameters.tokenPosition);
					drawPanel.updateDraw(true);
					frame.setStatusBarText(LaTeXDrawLang.getString1_6("LaTeXDrawFrame.36")); //$NON-NLS-1$
					frame.getLMenuBar().getExportAsMenu().setEnabled(true);
					frame.getToolbar().getButtonPdfExport().setEnabled(true);
					drawPanel.setIsModified(true);
				}
				break;
		}
		
		resetProgressBar();
	}


	
	@Override
	public void resetProgressBar()
	{
		super.resetProgressBar();
		
		codeToParse = null;
	}
	
	
	/**
	 * @return the codeToParse.
	 * @since 2.0.0
	 */
	public String getCodeToParse()
	{
		return codeToParse;
	}


	/**
	 * @param codeToParse the codeToParse to set.
	 * @since 2.0.0
	 */
	public void setCodeToParse(String codeToParse)
	{
		this.codeToParse = codeToParse;
	}
}
