/* 
 * Copyright (C) 2004, 2005 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <eel/eel.h>
#include <translate.h>
#include "gt-util.h"
#include "gt-shell.h"
#include "gt-language-view.h"

#define CONF_KEY			"gt-language-view-conf-key"
#define PAIR_FLAGS			"gt-language-view-pair-flags"

enum {
  COLUMN_TAG,
  COLUMN_NAME,
  N_COLUMNS
};

enum {
  SOURCE_COLUMN_DEST_STORE = N_COLUMNS,
  SOURCE_N_COLUMNS
};

static void gt_language_view_source_changed_h (GtkTreeSelection *selection,
					       gpointer user_data);
static void gt_language_view_dest_changed_h (GtkTreeSelection *selection,
					     gpointer user_data);

static void gt_language_view_setup (GtkTreeView *view,
				    const char *conf_key,
				    TranslatePairFlags pair_flags,
				    GCallback selection_changed_handler);
static void gt_language_view_source_fill (GtkTreeView *view);

char *gt_language_view_get_conf_selected_tag (GtkTreeView *view);
void gt_language_view_set_conf_selected_tag (GtkTreeView *view, const char *tag);

static gboolean gt_language_view_model_get_iter (GtkTreeModel *model,
						 const char *tag,
						 GtkTreeIter *iter);

void
gt_language_view_setup_pair (GtkTreeView *source_view,
			     const char *source_conf_key,
			     GtkTreeView *dest_view,
			     const char *dest_conf_key,
			     TranslatePairFlags pair_flags,
			     GCallback selection_changed_handler)
{
  GtkTreeSelection *selection;
  
  g_return_if_fail(GTK_IS_TREE_VIEW(source_view));
  g_return_if_fail(source_conf_key != NULL);
  g_return_if_fail(GTK_IS_TREE_VIEW(dest_view));
  g_return_if_fail(dest_conf_key != NULL);
  g_return_if_fail(selection_changed_handler != NULL);

  gt_language_view_setup(source_view, source_conf_key, pair_flags, selection_changed_handler);
  gt_language_view_setup(dest_view, dest_conf_key, pair_flags, selection_changed_handler);

  gt_g_object_connect(source_view, gt_shell_get_translate_session(gt_shell),
		      "swapped-signal::notify::pairs", gt_language_view_source_fill, source_view,
		      NULL);

  selection = gtk_tree_view_get_selection(source_view);
  g_signal_connect(selection, "changed", G_CALLBACK(gt_language_view_source_changed_h), dest_view);

  selection = gtk_tree_view_get_selection(dest_view);
  g_signal_connect(selection, "changed", G_CALLBACK(gt_language_view_dest_changed_h), dest_view);

  gt_language_view_source_fill(source_view);
}

static void
gt_language_view_source_changed_h (GtkTreeSelection *selection,
				   gpointer user_data)
{
  GtkTreeView *source_view = gtk_tree_selection_get_tree_view(selection);
  GtkTreeView *dest_view = user_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkListStore *dest_store = NULL;
  char *from = NULL;
  char *to;

  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    gtk_tree_model_get(model, &iter,
		       COLUMN_TAG, &from,
		       SOURCE_COLUMN_DEST_STORE, &dest_store,
		       -1);

  gt_language_view_set_conf_selected_tag(source_view, from);
  g_free(from);

  gtk_tree_view_set_model(dest_view, (GtkTreeModel *) dest_store);
  if (dest_store)
    g_object_unref(dest_store);

  gtk_tree_view_set_search_column(dest_view, COLUMN_NAME);

  to = gt_language_view_get_conf_selected_tag(dest_view);
  gt_language_view_select(dest_view, to);
  g_free(to);
}

static void
gt_language_view_dest_changed_h (GtkTreeSelection *selection,
				 gpointer user_data)
{
  GtkTreeView *dest_view = user_data;
  char *to;

  to = gt_language_view_get_selected(dest_view);
  gt_language_view_set_conf_selected_tag(dest_view, to);
  g_free(to);
}

static void
gt_language_view_setup (GtkTreeView *view,
			const char *conf_key,
			TranslatePairFlags pair_flags,
			GCallback selection_changed_handler)
{
  GtkTreeViewColumn *column;
  GtkTreeSelection *selection;

  g_return_if_fail(GTK_IS_TREE_VIEW(view));
  g_return_if_fail(conf_key != NULL);
  g_return_if_fail(selection_changed_handler != NULL);

  column = gtk_tree_view_column_new_with_attributes(NULL,
						    gtk_cell_renderer_text_new(),
						    "text", COLUMN_NAME,
						    NULL);
  gtk_tree_view_append_column(view, column);

  g_object_set_data_full(G_OBJECT(view), CONF_KEY, g_strdup(conf_key), g_free);
  g_object_set_data(G_OBJECT(view), PAIR_FLAGS, GUINT_TO_POINTER(pair_flags));

  selection = gtk_tree_view_get_selection(view);
  g_signal_connect(selection, "changed", selection_changed_handler, NULL);
}

static void
gt_language_view_source_fill (GtkTreeView *view)
{
  GtkListStore *source_store;
  GSList *pairs;
  GHashTable *dest_stores;
  GSList *l;
  char *selected_tag;
  TranslatePairFlags pair_flags;

  g_return_if_fail(GTK_IS_TREE_VIEW(view));

  pair_flags = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(view), PAIR_FLAGS));

  source_store = gtk_list_store_new(SOURCE_N_COLUMNS,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    GTK_TYPE_LIST_STORE);

  pairs = translate_session_get_pairs(gt_shell_get_translate_session(gt_shell));
  pairs = translate_pairs_sort_by_name(pairs);

  dest_stores = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, g_object_unref);

  GT_LIST_FOREACH(l, pairs)
    {
      TranslatePair *pair = l->data;
      const char *from = translate_pair_get_from(pair);
      const char *to = translate_pair_get_to(pair);
      GtkTreeIter iter;
      GtkListStore *dest_store;

      if (! (translate_pair_get_flags(pair) & pair_flags))
	continue;

      dest_store = g_hash_table_lookup(dest_stores, from);
      if (! dest_store)
	{
	  dest_store = gtk_list_store_new(N_COLUMNS,
					  G_TYPE_STRING,
					  G_TYPE_STRING);

	  gtk_list_store_append(source_store, &iter);
	  gtk_list_store_set(source_store, &iter,
			     COLUMN_TAG, from,
			     COLUMN_NAME, translate_get_language_name(from),
			     SOURCE_COLUMN_DEST_STORE, dest_store,
			     -1);

	  g_hash_table_insert(dest_stores, (gpointer) from, dest_store);
	}
      
      gtk_list_store_append(dest_store, &iter);
      gtk_list_store_set(dest_store, &iter,
			 COLUMN_TAG, to,
			 COLUMN_NAME, translate_get_language_name(to),
			 -1);
    }

  gt_g_object_slist_free(pairs);
  g_hash_table_destroy(dest_stores);

  gtk_tree_view_set_model(view, GTK_TREE_MODEL(source_store));
  g_object_unref(source_store);

  gtk_tree_view_set_search_column(view, COLUMN_NAME);

  selected_tag = gt_language_view_get_conf_selected_tag(view);
  gt_language_view_select(view, selected_tag);
  g_free(selected_tag);
}

char *
gt_language_view_get_conf_selected_tag (GtkTreeView *view)
{
  const char *conf_key;

  g_return_val_if_fail(GTK_IS_TREE_VIEW(view), NULL);

  conf_key = g_object_get_data(G_OBJECT(view), CONF_KEY);
  g_return_val_if_fail(conf_key != NULL, NULL);

  return eel_gconf_get_string(conf_key);
}

void
gt_language_view_set_conf_selected_tag (GtkTreeView *view, const char *tag)
{
  const char *conf_key;

  g_return_if_fail(GTK_IS_TREE_VIEW(view));

  conf_key = g_object_get_data(G_OBJECT(view), CONF_KEY);
  g_return_if_fail(conf_key != NULL);

  if (tag)
    eel_gconf_set_string(conf_key, tag);
  else
    eel_gconf_unset(conf_key);
}

char *
gt_language_view_get_selected (GtkTreeView *view)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  char *tag = NULL;

  g_return_val_if_fail(GTK_IS_TREE_VIEW(view), NULL);

  if (gtk_tree_selection_get_selected(gtk_tree_view_get_selection(view), &model, &iter))
    gtk_tree_model_get(model, &iter, COLUMN_TAG, &tag, -1);

  return tag;
}

gboolean
gt_language_view_has_selected (GtkTreeView *view)
{
  g_return_val_if_fail(GTK_IS_TREE_VIEW(view), FALSE);

  return gtk_tree_selection_get_selected(gtk_tree_view_get_selection(view), NULL, NULL);
}

static gboolean
gt_language_view_model_get_iter (GtkTreeModel *model,
				 const char *tag,
				 GtkTreeIter *iter)
{
  gboolean valid;
  GtkTreeIter _iter;

  g_return_val_if_fail(GTK_IS_TREE_MODEL(model), FALSE);

  if (! tag)
    return FALSE;

  valid = gtk_tree_model_get_iter_first(model, &_iter);
  while (valid)
    {
      char *this_tag;
      gboolean has;

      gtk_tree_model_get(model, &_iter, COLUMN_TAG, &this_tag, -1);
      has = ! g_ascii_strcasecmp(this_tag, tag);
      g_free(this_tag);

      if (has)
	{
	  if (iter)
	    *iter = _iter;

	  return TRUE;
	}

      valid = gtk_tree_model_iter_next(model, &_iter);
    }

  return FALSE;
}

void
gt_language_view_select (GtkTreeView *view, const char *tag)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *path;

  g_return_if_fail(GTK_IS_TREE_VIEW(view));

  selection = gtk_tree_view_get_selection(view);

  model = gtk_tree_view_get_model(view);
  if (! model)
    return;

  if (! gt_language_view_model_get_iter(model, tag, &iter))
    {
      if (! gtk_tree_model_get_iter_first(model, &iter))
	return;
    }

  gtk_tree_selection_select_iter(selection, &iter);

  path = gtk_tree_model_get_path(model, &iter);
  gtk_tree_view_scroll_to_cell(view, path, NULL, TRUE, 0.5, 0);
  gtk_tree_path_free(path);
}
