/***************************************************************************
                              kstviewarrow.cpp
                             -------------------
    begin                : Jun 14, 2005
    copyright            : (C) 2005 The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>

#include <klocale.h>

#include <qmetaobject.h>
#include <qpainter.h>
#include <qvariant.h>

#include "kstviewarrow.h"

KstViewArrow::KstViewArrow()
: KstViewLine("Arrow") {
  _fromArrowScaling = 1;
  _toArrowScaling = 1;
}


KstViewArrow::KstViewArrow(const QDomElement& e)
: KstViewLine(e) {

  int orientationInt = 0;
  
  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement el = n.toElement(); 
    if (!el.isNull()) {
      if (el.tagName() == "orientation") {
        orientationInt = el.text().toInt();
      } else if (metaObject()->findProperty(el.tagName().latin1(), true) > -1) {
        setProperty(el.tagName().latin1(), QVariant(el.text()));  
      }  
    }
    n = n.nextSibling();      
  }
  
  switch (orientationInt) {
    case 0:
      _orientation = UpLeft;
      break;
    case 1:
      _orientation = UpRight;
      break;
    case 2:
      _orientation = DownLeft;
      break;
    case 3:
      _orientation = DownRight;
      break;
    default:
      _orientation = UpLeft;  
  }
 
  // always has this value
  _type = "Arrow";
}


KstViewArrow::~KstViewArrow() {
}


void KstViewArrow::paint(KstPaintType type, QPainter& p, const QRegion& bounds) {
  if (hasArrow()) {
    int w = width();
    QPen pen(_foregroundColor, w);
    pen.setCapStyle(capStyle());
    p.setPen(pen);
    p.setBrush(_foregroundColor);
    QPointArray pts(3);
    if (_hasToArrow) {
      QPoint to = KstViewLine::to();
      QPoint from = KstViewLine::from();    
      double rise = double(from.y() - to.y());
      double run = double(from.x() - to.x());
      double theta = 0.0;
      if (run == 0) {
        if (rise < 0) {
          theta = 0.5*M_PI;
        }
        else {
          theta = 1.5*M_PI;
        }
      }
      else {
        theta = atan(rise/run);
        if (run > 0) {
          theta += M_PI;  
        }
        if (rise > 0) {
          theta += 2*M_PI;  
        }
      }
 
      double sina = sin(theta + M_PI / 2);
      double cosa = cos(theta + M_PI / 2);
      int l = (int)(_toArrowScaling * 2.0 * w);
      pts[0] = to;
      pts[1] = QPoint(to.x() + l, to.y() + int(sqrt(3) * l));
      pts[2] = QPoint(to.x() - l, to.y() + int(sqrt(3) * l));
      QWMatrix m(cosa, sina, -sina, cosa, 0, 0);
      pts.translate(-to.x(), -to.y());
      pts = m.map(pts);
      pts.translate(to.x(), to.y());
      p.drawPolygon(pts);
    }
    if (_hasFromArrow) {
      QPoint to = KstViewLine::from();
      QPoint from = KstViewLine::to();    
      double rise = double(from.y() - to.y());
      double run = double(from.x() - to.x());
      double theta = 0.0;
      if (run == 0) {
        if (rise < 0) {
          theta = 0.5*M_PI;
        }
        else {
          theta = 1.5*M_PI;
        }
      }
      else {
        theta = atan(rise/run);
        if (run > 0) {
          theta += M_PI;  
        }
        if (rise > 0) {
          theta += 2*M_PI;  
        }
      }
      
      double sina = sin(theta + M_PI / 2);
      double cosa = cos(theta + M_PI / 2);
      int l = (int)(_fromArrowScaling * 2.0 * w);
      pts[0] = to;
      pts[1] = QPoint(to.x() + l, to.y() + int(sqrt(3) * l));
      pts[2] = QPoint(to.x() - l, to.y() + int(sqrt(3) * l));
      QWMatrix m(cosa, sina, -sina, cosa, 0, 0);
      pts.translate(-to.x(), -to.y());
      pts = m.map(pts);
      pts.translate(to.x(), to.y());
      p.drawPolygon(pts);
    }
  }
  KstViewLine::paint(type, p, bounds);
}


void KstViewArrow::save(QTextStream& ts, const QString& indent) {
  ts << indent << "<" << type() << ">" << endl;
  ts << indent + "  " << "<orientation>" << _orientation << "</orientation>" << endl;
  KstViewObject::save(ts, indent + "  ");
  ts << indent << "</" << type() << ">" << endl;
}


bool KstViewArrow::hasArrow() const {
  return _hasToArrow || _hasFromArrow;
}


QMap<QString, QVariant> KstViewArrow::widgetHints(const QString& propertyName) const {
  QMap<QString, QVariant> map = KstViewLine::widgetHints(propertyName);
  if (!map.empty()) {
    return map;  
  }
  if (propertyName == "hasFromArrow") {
    map.insert(QString("_kst_widgetType"), QString("QCheckBox"));
    map.insert(QString("_kst_label"), QString::null);
    map.insert(QString("text"), i18n("Arrow at start"));  
  } else if (propertyName == "hasToArrow") {
    map.insert(QString("_kst_widgetType"), QString("QCheckBox"));
    map.insert(QString("_kst_label"), QString::null);    
    map.insert(QString("text"), i18n("Arrow at end"));
  } else if (propertyName == "fromArrowScaling") {
    map.insert(QString("_kst_widgetType"), QString("KDoubleSpinBox"));
    map.insert(QString("_kst_label"), i18n("Start arrow scaling")); 
    map.insert(QString("minValue"), 1.0);
    map.insert(QString("maxValue"), 100.0);
  } else if (propertyName == "toArrowScaling") {
    map.insert(QString("_kst_widgetType"), QString("KDoubleSpinBox"));
    map.insert(QString("_kst_label"), i18n("End arrow scaling")); 
    map.insert(QString("minValue"), 1.0);
    map.insert(QString("maxValue"), 100.0);
  }
  return map;
}


bool KstViewArrow::hasFromArrow() const {
  return _hasFromArrow;  
}


void KstViewArrow::setHasFromArrow(bool yes) {
  _hasFromArrow = yes;  
}


bool KstViewArrow::hasToArrow() const {
  return _hasToArrow;  
}


void KstViewArrow::setHasToArrow(bool yes) {
  _hasToArrow = yes;  
}


double KstViewArrow::fromArrowScaling() const {
  return _fromArrowScaling;  
}


void KstViewArrow::setFromArrowScaling(double scaling) {
  if (scaling < 1.0) {
    scaling = 1.0;  
  }  
  _fromArrowScaling = scaling;
}


double KstViewArrow::toArrowScaling() const {
  return _toArrowScaling;  
}


void KstViewArrow::setToArrowScaling(double scaling) {
  if (scaling < 1.0) {
    scaling = 1.0;  
  }  
  _toArrowScaling = scaling;
}


#include "kstviewarrow.moc"
// vim: ts=2 sw=2 et
