// This file is part of KLyX, The Document Processor
//
// Copyright (C) 1995 Matthias Ettrich
// Copyright (C) 1997-1999 KLyX Team


#include "klyx.h"
#include "LyXView.h"
#include "tex-strings.h"

#include <klocale.h>
#include <kcharsets.h>

#include <qscrbar.h>
#include <qtimer.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#define XK_MISCELLANY
#define XK_XKB_KEYS
#include <X11/keysymdef.h>

#include <fcntl.h>



extern int GhostscriptMsg(XEvent *ev);
Window gs_communication_window;
Display* qt_display;
extern void QuitLyX();

static K_Canvas *focus_canvas = 0;
static K_Canvas *last_focus_canvas = 0;
static XEvent stored_key_event;


LString DEFCHSET_VAR = DEFCHSET_PRIM;
LString DEFENCODING_VAR = DEFENCODING_PRIM;
LString DEFLANG_VAR = DEFLANG_PRIM;
LString DEFLANGUAGE_VAR = DEFLANGUAGE_PRIM;



static QList<KLyXToplevelWidget> memberList;



void deleteAllToplevelWidgets()
{
    while( memberList.count() )
	delete memberList.first();
}



KLyXToplevelWidget::KLyXToplevelWidget(LyXView* _view, const char* name)
    : KTMainWindow( name ),
      view( _view )
{
    ::memberList.append( this );
}



KLyXToplevelWidget::~KLyXToplevelWidget()
{
    delete view;
    ::memberList.removeRef( this );
}



bool KLyXToplevelWidget::queryExit(){
  QuitLyX();
  return false;
}

K_Canvas::K_Canvas (BufferView* _view, QWidget *parent=0, const char* name=0): 
  QWidget(parent, name)
{
  XSetWindowAttributes attributes;
  unsigned long valuemask;

  view = _view;

  work_area = new QRect;

  // Matthias: it is better to be slower instead of flickering
  valuemask = CWBackingStore;
  attributes.backing_store = Always;
  
  XChangeWindowAttributes(qt_xdisplay(), winId(),
 			  valuemask, &attributes);

  setBackgroundMode(NoBackground);
  
  setCursor(ibeamCursor);

  scrollbar = new QScrollBar(this);
  scrollbar->setCursor(arrowCursor);
  connect(scrollbar, SIGNAL(valueChanged(int)), SLOT(scrollbarChanged(int)));

  updateTimer = new QTimer(this);
  connect(updateTimer, SIGNAL(timeout()), SLOT(updateTimerTimeout()));
  startTimer(400);
  setFocusPolicy(StrongFocus);
  if (hasFocus())
    focus_canvas = this;

  
}

void K_Canvas::takeFocus(){
  setFocus();
  focusInEvent(0);
}


K_Canvas::~K_Canvas(){
  if (focus_canvas == this)
    focus_canvas = 0;
  if (last_focus_canvas == this)
    last_focus_canvas = 0;
  delete work_area;
};

void K_Canvas::startUpdateTimer(float timer){
  updateTimer->start(int(timer*1000), TRUE);
}
void K_Canvas::stopUpdateTimer(){
  updateTimer->stop();
}



void K_Canvas::scrollbarChanged(int value){
  if (view)
    view->kScrollCB(value);
}

void K_Canvas::updateTimerTimeout(){
  if (view)
    view->updateTimerTimeout();

}



void K_Canvas::timerEvent( QTimerEvent * ){
  if (view && focus_canvas == this)
    view->kCursorToggleCB();
}

void K_Canvas::mousePressEvent( QMouseEvent *e ){
  int b = 1;
  if (e->button() == MidButton) b = 2;
  if (e->button() == RightButton) b = 3;
  if (view)
    view->kButtonPress(e->pos().x(), e->pos().y(), b);
}
void K_Canvas::mouseReleaseEvent( QMouseEvent *e ){
  int b = 1;
  if (e->button() == MidButton) b = 2;
  if (e->button() == RightButton) b = 3;
  if (view)
    view->kButtonRelease(e->pos().x(), e->pos().y(), b);
}
void K_Canvas::mouseDoubleClickEvent( QMouseEvent *e){
  int b = 1;
  if (e->button() == MidButton) b = 2;
  if (e->button() == RightButton) b = 3;
  if (view)
    view->kButtonDoubleClick(e->pos().x(), e->pos().y(), b);
}
void K_Canvas::mouseMoveEvent( QMouseEvent *e ){
  if (view)
    view->kMouseMove(e->pos().x(), e->pos().y());
}
void K_Canvas::keyPressEvent( QKeyEvent *e ){
   if (view) {
     // THIS is really a horrible hack - LyX does not recognize some accented
     // chars properly, thinking that keycode is 0. So we store Qt-recognized
     // char into "serial" field of key event (about which nobody cares
     // anyway). If LyXFunc::processKeyEvent detects that keysym is zero,
     // it'll use this stored ascii code. The reason why it can detect zero
     // keysym is, that LyX doesn't use XIM extension yet, so it can't handle
     // keyboard mappings correctly. [schreter980507]
     stored_key_event.xkey.serial = e->ascii();
     view->kKeyPress(&stored_key_event);
   }
}
void K_Canvas::keyReleaseEvent( QKeyEvent * ){}
void K_Canvas::focusInEvent( QFocusEvent * ){
  if (view){
    if (last_focus_canvas == this)
      view->setBufferInternal(view->currentBuffer());
    else
      view->setBuffer(view->currentBuffer());
  }
  focus_canvas = this;
  last_focus_canvas = this;
}
void K_Canvas::focusOutEvent( QFocusEvent * ){
  if (view)
    view->kHideCursor();
  stopUpdateTimer();
  updateTimerTimeout();
  if (focus_canvas == this)
    focus_canvas = 0;
}

void K_Canvas::enterEvent( QEvent * ){}
void K_Canvas::leaveEvent( QEvent * ){}
void K_Canvas::paintEvent( QPaintEvent * ){
  if (!view || !view->getScreen()){
    XFillRectangle(qt_xdisplay(), winId(),
		   getGC(gc_lighted),
		   work_area->x(), 
		   work_area->y(),
		   work_area->width(),
		   work_area->height());
    // erase(*work_area); This does not work with NoBackground!
  }
  else {
    view->redraw();
  }
}
void K_Canvas::moveEvent( QMoveEvent * ){}
void K_Canvas::resizeEvent( QResizeEvent * ){
  scrollbar->setGeometry(width()-16, 0, 16, height());
  *work_area = QRect(0, 0, width()-16, height()); 
  if (view)
    view->resize();
}

K_Frame::K_Frame (QWidget *parent=0, const char* name=0): QWidget(parent, name){
  f1 = new QFrame(this);
  f1->setFrameStyle(QFrame::Panel|QFrame::Sunken);
  f2 = new QFrame(this);
  f2->setFrameStyle(QFrame::Panel|QFrame::Sunken);
}

void K_Frame::setWidgets(QWidget* _w, QWidget* _s){
  w = _w;
  s = _s;
  s->setFocusPolicy(ClickFocus);
  resizeEvent(0);
}


void K_Frame::resizeEvent( QResizeEvent * ){
    f1->setGeometry(0,0,width(),height()-26);
    f2->setGeometry(0, height()-24, width(), 24);
  
  if (w)
    w->setGeometry(f1->x()+2, f1->y()+2, f1->width()-4, f1->height()-4);
  if (s)
    s->setGeometry(f2->x()+2, f2->y()+2, f2->width()-4, f2->height()-4);
}



MyApp::MyApp(int &argc, char **argv , const QString& rAppName):
  KApplication(argc, argv, rAppName){
    QWidget* tmp = new QWidget;
    gs_communication_window = tmp->winId();

    qt_display = tmp->x11Display();

    // ensure that a fork()/exec() pair will not confuse X
    fcntl(ConnectionNumber(qt_xdisplay()), F_SETFD, 1);
    
    // find out what is the default character set (iso font norm)
    QString defchset = kapp->getLocale()->charset();
    KCharset chset(defchset);
    QString chs = chset.xCharset();
    if (chs.left(3) == "iso" || chs.left(4) == "koi8")
       DEFCHSET_VAR = chs;
    if (chs.left(8) == "iso8859-") {
       char buf[32];
       sprintf(buf, "latin%c", chs[8]);
       DEFENCODING_VAR = buf;
    }

    // now try to find out what language we are using and set certain
    // params accordingly
    DEFLANG_VAR = kapp->getLocale()->language();

    // find language name to language code
    for (int i = 0; tex_babel[i].name[0]; ++i) {
	if (DEFLANG_VAR == tex_babel[i].code) {
		DEFLANGUAGE_VAR = tex_babel[i].name;
		break;
	}
    }

	printf("Defchset: %s, defencoding: %s, deflang: %s/%s\n",
		DEFCHSET, DEFENCODING, DEFLANG, DEFLANGUAGE);
}

bool MyApp::x11EventFilter( XEvent * ev){
  if (ev->type == KeyPress){
    stored_key_event = *ev;
    XKeyEvent *keyevent = &ev->xkey;
    XComposeStatus compose_status={NULL,0};
    KeySym keysym_return;
    char s_r[10];
    int num_bytes;
    num_bytes = XLookupString(keyevent,
			      s_r,
			      10,
			      &keysym_return,
			      &compose_status);
    // do some tricky stuff because LyX wants to handle dead-keys manually
    // although Qt would do that already for us. 
    if (focus_canvas && focus_canvas->hasFocus() && 
	focus_canvas->view &&
	(false ||
#ifdef   XK_dead_grave
	 keysym_return == XK_dead_grave ||
#endif
#ifdef	 XK_dead_acute
	 keysym_return == XK_dead_acute ||
#endif
#ifdef	 XK_dead_circumflex
	 keysym_return == XK_dead_circumflex ||
#endif
#ifdef	 XK_dead_tilde
	 keysym_return == XK_dead_tilde ||
#endif
#ifdef	 XK_dead_macron
	 keysym_return == XK_dead_macron ||
#endif
#ifdef	 XK_dead_breve
	 keysym_return == XK_dead_breve ||
#endif
#ifdef	 XK_dead_abovedot
	 keysym_return == XK_dead_abovedot ||
#endif
#ifdef	 XK_dead_diaeresis
	 keysym_return == XK_dead_diaeresis ||
#endif
#ifdef	 XK_dead_abovering
	 keysym_return == XK_dead_abovering ||
#endif
#ifdef	 XK_dead_doubleacute
	 keysym_return == XK_dead_doubleacute ||
#endif
#ifdef	 XK_dead_caron
	 keysym_return == XK_dead_caron ||
#endif
#ifdef	 XK_dead_cedilla
	 keysym_return == XK_dead_cedilla ||
#endif
#ifdef	 XK_dead_ogonek
	 keysym_return == XK_dead_ogonek ||
#endif
#ifdef	 XK_dead_iota 
	 keysym_return == XK_dead_iota  ||
#endif
#ifdef	 XK_dead_voiced_sound
	 keysym_return == XK_dead_voiced_sound ||
#endif
#ifdef	 XK_dead_semivoiced_sound
	 keysym_return == XK_dead_semivoiced_sound ||
#endif
#ifdef	 XK_dead_belowdot             
	 keysym_return == XK_dead_belowdot ||
#endif
	 false)){
      stored_key_event.xkey.serial = 0;
	// see comment in K_Canvas::keyPressEvent
      focus_canvas->view->kKeyPress(&stored_key_event);
       return TRUE;
    }
  }
  else if (ev->type == ClientMessage 
	   && ev->xany.window == gs_communication_window){
    GhostscriptMsg(ev);
    return TRUE;
    
  }
  return KApplication::x11EventFilter(ev);
}



// Local Variables:
// mode: C++
// c-file-style: "Stroustrup"
// End:
