/* Copyright (C) 1999 Claus Wilke <claus.wilke@gmx.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qmultilineedit.h>
#include <qgroupbox.h>
#include <qcombobox.h>
#include <qdom.h>
#include <stdio.h>
#include <kdialogbase.h>
#include <kmessagebox.h>
#include <kseparator.h>
#include <klocale.h>

#include "bibentrydialog.h"
#include "bibentryfieldviews.h"
#include "entrymanipulator.h"
#include "kbibliodoc.h"
#include "kbiblioshell.h"

BibEntryView::BibEntryView( QDomElement entry, KBiblioShell *shell, QWidget *parent, const char *name )
    : QWidget( parent, name )
{

    // Make the top-level layout; a vertical box to contain all widgets
    // and sub-layouts.

    mShell = shell;
    mEntry = entry;
    mManip = new EntryManipulator( mEntry, mShell );

    mTopLayout = new QVBoxLayout( this, 0, 0 );

    QBoxLayout *hbox = new QHBoxLayout( mTopLayout );
    QLabel *label = new QLabel( i18n( "Label:" ), this );
    hbox->addWidget( label );
    hbox->addSpacing( mShell->spacingHint() );
    mLined = new QLineEdit( this );
    mLined->setText( mManip->label() );
    mLined->setMinimumWidth( 25*fontMetrics().width( "x" ) );
    connect( mLined, SIGNAL( textChanged( const QString & ) ), this, SLOT( setEditedTrue() ) );
    hbox->addWidget( mLined );

    hbox->addSpacing( mShell->spacingHint() );

    label = new QLabel( i18n( "Type:" ), this );
    hbox->addWidget( label );
    hbox->addSpacing( mShell->spacingHint() );

    mTypeCombo = new QComboBox( this );
    mEntryTypes = mShell->document()->entryTypesNames();
    mOldIndex = mEntryTypes.findIndex( mManip->name() );
    if ( mOldIndex < 0 ){
	mOldIndex = mEntryTypes.count();
	mEntryTypes.append( mManip->name() );
    }
    mTypeCombo->insertStringList( mShell->entryDescriptionList( mEntryTypes ) );
    mTypeCombo->setCurrentItem( mOldIndex );
    mTypeCombo->setMaximumSize( mTypeCombo->sizeHint() );
    connect( mTypeCombo, SIGNAL( activated( int ) ), this, SLOT( typeChanged( int ) ) );
    hbox->addWidget( mTypeCombo );

    mTopLayout->addSpacing( mShell->spacingHint() );
    mTopLayout->addWidget( new KSeparator( this ) );
    mTopLayout->addSpacing( mShell->spacingHint() );

    mGridLayout = new QGridLayout( mTopLayout, 0, 0, mShell->spacingHint() );

    QStringList editFieldNames = mShell->editableFields( mEntry.nodeName() );

    if ( !editFieldNames.isEmpty() )
	setupEditWidgets( editFieldNames );
    else
	setupEmptyWidget();

    mEdited = false;

    mTopLayout->activate();
}


BibEntryView::~BibEntryView()
{
    if ( mFieldViews!= 0)
	delete mFieldViews;
    delete mManip;
}


BibEntryFieldView*
BibEntryView::addEntryView( int row, const QString &fieldName )
{
    BibEntryFieldView *newView = 0;

    switch ( mManip->fieldDataType( fieldName ) ){
    case BibEntryFieldType::STRING:
	newView = new StringFieldView( mEntry, fieldName, mShell, mGridLayout, row, this );
	break;
    case BibEntryFieldType::TEXT:
	newView = new TextFieldView( mEntry, fieldName, mShell, mGridLayout, row, this );
	break;
    default:
	newView = 0;
    }

    if ( 0 != newView){
	mFieldViews->append( newView );
	connect ( newView, SIGNAL( hasChanged() ), this, SLOT( setEditedTrue() ) );
    }
    return newView;
}


void
BibEntryView::setupEditWidgets(const QStringList &editFieldNames )
{
    mFieldViews = new QList<BibEntryFieldView> ();
    mFieldViews->setAutoDelete( true );

    BibEntryFieldView *newView = 0, *oldView = 0;
    int row = 0;
    for ( QStringList::ConstIterator it = editFieldNames.begin(); it != editFieldNames.end(); ++it ) {
	oldView = newView;
	newView = addEntryView( row, (*it) );
	if ( 0 != newView ) {
	    if ( 0 != oldView )
		connect ( oldView, SIGNAL( returnPressed() ), newView, SLOT( setFocus() ) );
	
	    row += 1;
	}
	
    }
}


void
BibEntryView::setupEmptyWidget()
{
    mFieldViews = 0;
    QLabel *label = new QLabel( i18n( "No method available to edit the fields of this entry." ), this );
    mGridLayout->addWidget( label, 0, 0 );
}


bool
BibEntryView::edited()
{
    return mEdited;
}


void
BibEntryView::setEditedTrue()
{
    mEdited = true;
}


void
BibEntryView::storeData()
{
    BibEntryFieldView *view;

    if ( mEdited ){
	mManip->setLabel( mLined->text() );
	if ( mFieldViews != 0 )
	    for ( view = mFieldViews->first(); view != 0; view = mFieldViews->next() )
	    view->storeData();
    }
}


void
BibEntryView::typeChanged( int index )
{
    KMessageBox::sorry( 0, QString( i18n( "Changing entry types is not implemented yet." ) ) );
    mTypeCombo->setCurrentItem( mOldIndex );
}

//============================================================
//
//             class BibEntryDialog


BibEntryDialog::BibEntryDialog( QDomElement entry, KBiblioShell *shell, QWidget *parent, const char *name )
    : KDialogBase ( parent, name, true, i18n( "Bibentry Dialog" ) , KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::No, true )
    // Don't change to non-blocking dialog
    // as long as you don't know exactly what you are doing. The dialog doesn't
    // know how to handle a change in the entry that comes from somewhere else!
{
    mEntryView = new BibEntryView ( entry, shell, this );
    setMainWidget ( mEntryView );
}


BibEntryDialog::~BibEntryDialog()
{
}


bool
BibEntryDialog::edited()
{
    return mEntryView->edited();
}


void
BibEntryDialog::slotOk()
{
    mEntryView->storeData();
    accept();
}




#include "bibentrydialog.moc"








