#include <iostream.h>
#include <ctype.h>
#include <stdlib.h>
#include <qcolor.h>
#include <qpushbt.h>
#include <qbitmap.h>
#include <qdatetime.h>
#include "colorOpt.h"
#include "StringToken.h"
#include "ircHtml.h"
#include "ircDefine.h"
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include <qclipbrd.h>
#include "ircApp.h"

#define hand_width  16
#define hand_height 16

static unsigned char hand_bits[] = {
        0x00,0x00,0xfe,0x01,0x01,0x02,0x7e,0x04,0x08,0x08,0x70,0x08,0x08,0x08,0x70,
        0x14,0x08,0x22,0x30,0x41,0xc0,0x20,0x40,0x12,0x80,0x08,0x00,0x05,0x00,0x02,
        0x00,0x00};
static unsigned char hand_mask_bits[] = {
        0xfe,0x01,0xff,0x03,0xff,0x07,0xff,0x0f,0xfe,0x1f,0xf8,0x1f,0xfc,0x1f,0xf8,
        0x3f,0xfc,0x7f,0xf8,0xff,0xf0,0x7f,0xe0,0x3f,0xc0,0x1f,0x80,0x0f,0x00,0x07,
        0x00,0x02};


void IrcHtml::Init(const char* Name)
{
  oldVerMaxValue = 0;

  KConfig *config = ircapp->config;
  
  bgColor    = config->readColorEntry( "BgColor",   &white );
  textColor  = config->readColorEntry( "TextColor", &black );
  infoColor  = config->readColorEntry( "InfoColor", &blue  );
  msgColor   = config->readColorEntry( "MsgColor",  &red   );
  actionColor= config->readColorEntry( "ActionColor", &red   );
  errorColor = config->readColorEntry( "ErrorColor",  &red   );

  QString fs;
  fs = config->readEntry("TextFontSize");
  if ( !fs.isEmpty() ){
     textFSize = fs.toInt();
  }
  else
     textFSize = 4;
  fs = config->readEntry("MsgFontSize");
  if ( !fs.isEmpty() ){
     msgFSize = fs.toInt();
  }
  else
     msgFSize = 4;
  fs = config->readEntry("InfoFontSize");
  if ( !fs.isEmpty() ){
     infoFSize = fs.toInt();
  }
  else
     infoFSize = 4;
  fs = config->readEntry("ActionFontSize");
  if ( !fs.isEmpty() ){
     actionFSize = fs.toInt();
  }
  else
     actionFSize = 4;
  fs = config->readEntry("ErrorFontSize");
  if ( !fs.isEmpty() ){
     errorFSize = fs.toInt();
  }
  else
     errorFSize = 4;

  textFName = config->readEntry( "TextFont" );
  if ( textFName.isEmpty() )
     textFName = "times";
  
  msgFName = config->readEntry( "MsgFont" );
  if ( msgFName.isEmpty() )        
     msgFName = "times";

  infoFName = config->readEntry( "InfoFont" );
  if ( infoFName.isEmpty() )        
     infoFName = "times";

  actionFName = config->readEntry( "ActionFont" );
  if ( actionFName.isEmpty() )        
     actionFName = "times";

  errorFName = config->readEntry( "ErrorFont" );
  if ( errorFName.isEmpty() )        
     errorFName = "times";

  htmlw  = 0L;
  sbVer  = 0L;
  fontBase = 2;
  standardFont = "times";
  fixedFont    = "times";

  setLog = false;

  QString n;
  if (Name)
    n="Log"+QString(Name);
  else
    n="LogOutputPage";
  n=n.stripWhiteSpace();
  QString novalid="~!&$%";
  for (uint i=0;i<n.length();i++){
     if (novalid.find(n[i])>=0)
        n.replace(i,1,"");
  }
  
  QString dir=config->readEntry("LogDir", ircapp->keircHome+"/log");
  QString logFile=dir+"/"+n+".log";
  log.setName(logFile);
}

IrcHtml::IrcHtml(QWidget* parent, const char* name): 
  QWidget(parent, name)
{
#ifdef EDEBUG
  cout << "Konstruktor IrcHtml" << endl;
#endif
  Init(name);

  QBitmap cbits( hand_width, hand_height, hand_bits, true);
  QBitmap cmask( hand_width, hand_height, hand_mask_bits, true );
  QCursor handCursor( cbits, cmask, 0, 0 );

  window = new QFrame(this);
  window->setFrameStyle(QFrame::Panel | QFrame::Sunken);

  htmlw  = new KHTMLWidget(window);
  CHECK_PTR(htmlw);

  htmlw->setMarginWidth(1); 

  connect(htmlw, SIGNAL(textSelected(bool )), SLOT(slotTextSelected(bool )));
  htmlw->setStandardFont(standardFont);
  htmlw->setFixedFont   (fixedFont);  
  htmlw->setDefaultFontBase(fontBase);
  htmlw->setFocusPolicy( QWidget::StrongFocus );
  htmlw->setFocus(); 
  htmlw->setURLCursor(handCursor);
  htmlw->setUpdatesEnabled (true);
  htmlw->setDefaultBGColor (bgColor);
  htmlw->setDefaultTextColors( textColor, infoColor, infoColor );

  sbVer = new QScrollBar(0, 0, 12, htmlw->height(), 0,QScrollBar::Vertical,   window);

  sbHor    = 0L;
  //sbHor = new QScrollBar(0, 0, 24, htmlw->width(),  0,QScrollBar::Horizontal, window);

  connect(htmlw, SIGNAL(scrollVert(int )),  SLOT(slotScrollVert(int )));
  connect(sbVer, SIGNAL(valueChanged(int)), htmlw, SLOT(slotScrollVert(int)) );
  //connect(htmlw, SIGNAL(scrollHorz(int ) ), SLOT( slotScrollHorz( int ) ) );
  //connect(sbHor, SIGNAL(valueChanged(int)), htmlw, SLOT(slotScrollHorz(int)) );

  connect(htmlw, SIGNAL(documentChanged() ), SLOT( slotDocumentChanged() ) );
  connect(htmlw, SIGNAL(resized(const QSize & ) ),  SLOT( slotResized(const QSize &)));  
  
  connect(htmlw, SIGNAL(URLSelected(const char*, int)),
	  this,  SLOT  (slotURLSelected(const char*, int)));

  contentList.setAutoDelete(true);
  for (int i=0;i<4;i++){
     HtmlContent* hc;
     contentList.append(hc=new HtmlContent(htmlw, 75));
     connect(hc,  SIGNAL(signNextBuffer(const char*)),
	     this,SLOT  (slotNextBuffer(const char*)));
     if (i!=0){
        hc->setPrev(contentList.at(i-1));
	if (i==3)
	   contentList.at(0)->setPrev(hc);
     }
  }
  currentContent  = 0;

  bgsColor.sprintf("#%06x", bgColor.blue() | (bgColor.green()<<8) | (bgColor.red()<<16));
  contentList.first()->start(bgImage, bgsColor);
}

IrcHtml::~IrcHtml()
{
#ifdef EDEBUG
  cout << "Destruktor IrcHtml" << endl;
#endif
}

QColor  IrcHtml::getColor(int nr)
{
  if (nr>15)
     return black;
  return IrcColor::color[nr];
}

QString IrcHtml::htmlFilter(const char* Txt)
{
#ifdef EDEBUG
  cout << "IrcHtml::htmlFilter:"<<Txt<<endl;
#endif
  if (!Txt)
     return "";
  QString t=Txt;
  QString o="";
  int i=0;
  bool isBold      = 0;
  bool isInverse   = 0;
  bool isUnderline = 0;
  int  isColor     = 0;

  while (i<(int)t.length()){
    switch (t[i])
      {
      case '\n':
	o += "<BR>";
	break;
      case '<':
	o += "&lt;";
	break;
      case '>':
	o += "&gt;";
	break;	
      case '&':
	o += "&amp;";
	break;
      case ' ':
	if (i>0){
	  o += " &nbsp;";
	  for (; t[i+1]==' '; i++)
	    o += "&nbsp;";
	}
	break;
      case '\t':
	o += " &nbsp;";
        for (int j=0;j<7;j++)
          o += "&nbsp;";
	break;
      case 0x02:           // CTRL-B is bold
	isBold = !isBold;
	if (isBold){
	   if (isUnderline)
	      o += "</U>";
	   if (isInverse)
	      o += "</I>";
	   o += "<B>";
	   if (isInverse)
	      o += "<I>";
	   if (isUnderline)
	      o += "<U>";
	}
	else{
	   if (isUnderline)
	      o += "</U>";
	   if (isInverse)
	      o += "</I>";
	   o += "</B>";
	   if (isInverse)
	      o += "<I>";
	   if (isUnderline)
	      o += "<U>";
	}
	break;
      case 0x03:           // CTRL-K
	if (isUnderline)
	   o += "</U>";
	if (isInverse)
	   o += "</I>";
	if (isBold)
	   o += "</B>";
	if (isColor%2 && isColor!=0)
	  o += "</font >";

	isColor++;
	int fg, bg;
	fg = -1;
	bg = -1;
	if (++i<(int)t.length()){
	   int len;
	   QString s = t.mid(i, t.length()-i);
	   StringToken st(s);
	   s = st.nextDigit(" ,",&len);
	   if (s!=0L){
	      i += len-1;
	      fg = s.toInt();
	      s = st.nextDigit(" ,",&len);
	      if (s!=0L){
		bg = s.toInt();
		i += len;
	      }

	      QString sc,sp(1024);
	      QColor  mColor = getColor(fg);
	      sc.sprintf("%06x", mColor.blue() | (mColor.green()<<8) | (mColor.red()<<16));
	      sp.sprintf("<font color=#%s size=%d face=%s>", sc.data(), fontSize, fontName.data());
	      if (bg>=0){
		 mColor = getColor(bg);
		 sc.sprintf("%06x", mColor.blue() | (mColor.green()<<8) | (mColor.red()<<16));
		 sp += "<td bgcolor=#"+sc+">";
	      }
	      o += sp;
	      if (isBold)
		 o += "<B>";
	      if (isInverse)
		 o += "<I>";
	      if (isUnderline)
		 o += "<U>";
	   }
	   else
	      --i;
	}
	else
	   --i;
	break;
      case 0x16:           //CTRL-V is inverse
	isInverse = !isInverse;
	if (isInverse){
	   if (isUnderline)
	      o += "</U>";
	   o += "<I>";
	   if (isUnderline)
	      o += "<U>";
	}
	else{
	   if (isUnderline)
	      o += "</U>";
	   o += "</I>";
	   if (isUnderline)
	      o += "<U>";
	}
	break;
      case 0x1F:           // CTRL-_ toggles underline on/off
	isUnderline = !isUnderline;
	if (isUnderline)
	  o += "<U>";
	else
	  o += "</U>";
	break;
      default:
	o += t[i];
      }
    i++;
  }
#ifdef EDEBUG
  cout << "IrcHtml::htmlFilter:...Ok:"<<o<<endl;
#endif

  if (isBold)
     o += "</B>";
  if (isUnderline)
     o += "</U>";
  if (isInverse)
     o += "</I>";
  if (isColor%2)
     o += "</font>";

  return o;
}

// Type: 
void IrcHtml::slotWrite(int Type, const char* Txt, bool Parsen)
{
#ifdef EDEBUG
  cout << "IrcHtml::slotWrite:"<<Txt<<endl;
#endif

  fontSize=fontBase;

  if (Type&TYPE_MSG){
     color   = msgColor;
     fontSize= msgFSize;
     fontName= msgFName;
  }
  else if (Type&TYPE_TEXT){
     color   = textColor;
     fontSize= textFSize;
     fontName= textFName;
  }
  else if (Type&TYPE_INFO){
     color   = infoColor;
     fontSize= infoFSize;
     fontName= infoFName;
  }
  else if (Type&TYPE_ACTION){
     color   = actionColor;
     fontSize= actionFSize;
     fontName= actionFName;
  }
  else if (Type&TYPE_ERROR){
     color   = errorColor;
     fontSize= errorFSize;
     fontName= errorFName;
  }
  else {
     color   = msgColor;
     fontSize= msgFSize;
     fontName= msgFName;
  }


  QString t=Txt;
  QString vor="";
  QString a,b,c;

  if (Parsen)
     t=htmlFilter(Txt);

  int idx;
  if ((idx=t.find("http:", 0, false))!=-1){
     a=t.left(idx)+" ";
     b=t.mid(idx, t.length()-idx);
     StringToken token(b);
     QString ht=token.nextToken();
     QString en=token.toEnd();
     c=a+"<a href=\""+ht+"\">"+ht+"</a> "+en;
  }
  else{
     c=t;
  }

  QString sc;
  QString img ="";
  QString font="";

  QString imgDir = ircapp->home+"/.keirc/";
  if (Type&TYPE_RAW)
     ;
  else if (Type&TYPE_ERROR && Type&TYPE_IMG)
     img="<img SRC=\"file:"+imgDir+"error.xpm\">";
  else if (Type&TYPE_INFO && Type&TYPE_IMG)
     img="<img SRC=\"file:"+imgDir+"info.xpm\">";
  else if (Type&TYPE_MSG && Type&TYPE_IMG)
    img="<img SRC=\"file:"+imgDir+"msg.xpm\">";
  else if (Type&TYPE_ACTION && Type&TYPE_IMG)
    img="<img SRC=\"file:"+imgDir+"action.xpm\">";
  else if (Type&TYPE_TCL)
    img="<img SRC=\"file:"+imgDir+"tcl.xpm\">";

  sc.sprintf("%06x", color.blue() | (color.green()<<8) | (color.red()<<16));  
  QString p;
  p.resize(c.length()+1024);
  p.sprintf("<table cellspacing=1 cellpadding=0><td><font color=#%s size=%d face=%s>",
	    sc.data(), fontSize, fontName.data()); 
  if (img.isEmpty())
     p +=c;
  else
     p +=img+" "+c;
  p +="</font></table>";

  if (setLog==true){
     QDateTime tm=QDateTime::currentDateTime();  
     QString sLog = "["+tm.toString()+"] "+QString(Txt)+"\n";
     if (log.writeBlock(sLog, sLog.length())<0){
        setLog = false;
	log.close();
     }
  }
  if (contentList.at()==currentContent)
     contentList.current()->write(p);
  else
     contentList.current()->writeBuffer(p);
#ifdef EDEBUG
  cout << "IrcHtml::slotWrite: Ok" << endl;
#endif
}

void IrcHtml::slotTextSelected( bool )
{
  QString s(1024);
  htmlw->getSelectedText(s);
  //s[1023]='\0';
  if (s.isEmpty())
     return;
  //XSetSelectionOwner(qt_xdisplay(), XA_PRIMARY, handle(), CurrentTime );
  QClipboard* cb = ircapp->clipboard();
  cb->setText(s);
}
    
void IrcHtml::slotScrollVert( int y )
{
  sbVer->setValue( y );
}    

void IrcHtml::slotScrollHorz( int x )
{
  if (sbHor!=0L)
     sbHor->setValue( x );
} 

void IrcHtml::slotDocumentChanged()
{
#ifdef EDEBUG
  cout << "IrcHtml::slotDocumentChanged()"<<endl;
#endif
  if (htmlw->docHeight() > htmlw->height() )
     sbVer->setRange( 0, htmlw->docHeight() - htmlw->height() );
  else
     sbVer->setRange( 0, 0 );

  
  if ( !sbVer->draggingSlider() && oldVerMaxValue==sbVer->value())
     sbVer->setValue ( sbVer->maxValue() );
  oldVerMaxValue = sbVer->maxValue();
}        

void IrcHtml::slotResized( const QSize & )
{
#ifdef EDEBUG
  cout << "IrcHtml::slotResized()"<<endl;
#endif 
    sbVer->setSteps( 12, htmlw->height() - 20 );

    if (htmlw->docHeight() > htmlw->height() )
        sbVer->setRange( 0, htmlw->docHeight() - htmlw->height() );
    else
        sbVer->setRange( 0, 0 );

}        

void IrcHtml::slotSetColor(const QColor &col, int Type)
{
  if (Type&TYPE_TEXT)
     textColor = col;
  if (Type&TYPE_MSG)
     msgColor = col;
  if (Type&TYPE_ACTION)
     actionColor = col;
  if (Type&TYPE_INFO)
     infoColor = col;
  if (Type&TYPE_ERROR)
     errorColor = col;
  if (Type&TYPE_BG){
     bgImage="";
     ircapp->writeEntry("BackgroundImage", bgImage);
     bgColor = col;
     bgsColor.sprintf("#%06x", bgColor.blue() | (bgColor.green()<<8) | (bgColor.red()<<16));
     contentList.current()->setBgColor(bgsColor);
     sbVer->setValue ( sbVer->maxValue() );
  }
}

void IrcHtml::slotSetBgImage(const char* url)
{
  bgsColor="";
  bgImage=url;
  contentList.current()->setBgImage(bgImage);
  sbVer->setValue ( sbVer->maxValue() );
}

void IrcHtml::slotSetFontSize(int Size, int Type)
{
  if (Type&TYPE_TEXT)
     textFSize = Size;
  if (Type&TYPE_MSG)
     msgFSize = Size;
  if (Type&TYPE_ACTION)
     actionFSize = Size;
  if (Type&TYPE_INFO)
     infoFSize = Size;
  if (Type&TYPE_ERROR)
     errorFSize = Size;
}

void IrcHtml::slotSetFontName(const char* Name, int Type)
{
  if (Type&TYPE_TEXT)
     textFName = Name;
  if (Type&TYPE_MSG)
     msgFName = Name;
  if (Type&TYPE_ACTION)
     actionFName = Name;
  if (Type&TYPE_INFO)
     infoFName = Name;
  if (Type&TYPE_ERROR)
     errorFName = Name;
}

void IrcHtml::slotSetFlag(int Type, bool state)
{
#ifdef EDEBUG
  cout << "IrcHtml::slotSetFlag:"<< Type<<":" <<state<<endl;
#endif
  if (Type&FLAG_LOG){
    if (state==true && setLog==false){
       setLog = log.open(IO_ReadWrite|IO_Append);
    }
    else if (state==false && setLog==true){
       log.flush();
       log.close();
       setLog=false;
    }
  }
  else if (Type&FLAG_CLEAR){
    htmlw->begin();
    htmlw->parse();
  }
#ifdef EDEBUG
  cout << "IrcHtml::slotSetFlag:"<< Type<<":" <<state<<"...Ok"<<endl;
#endif
}

void IrcHtml::slotURLSelected(const char *_url, int _button)
{
#ifdef EDEBUG
  cout << "IrcHtml::slotURLSelected:"<<_url<<endl;
#endif
  if (_button==LeftButton){
     ircapp->browse("openURL", _url);
  }
}

void IrcHtml::slotNextBuffer(const char* txt)
{
  HtmlContent* hc=contentList.next();
  if (!hc)
     hc = contentList.first();
  currentContent =contentList.at();
  hc->start(bgImage, bgsColor);
  hc->write(txt);
  // .........
  sbVer->setValue ( sbVer->maxValue() );
}

void IrcHtml::slotPrevBuffer(const char*)
{
}

void IrcHtml::slotBackward()
{
  int actual = contentList.at();
  int tmp=currentContent-actual;
  if ( tmp==1 || (tmp+contentList.count())==1)
     return;
  if (--currentContent<0)
     currentContent=contentList.count()-1;

  HtmlContent* hc=contentList.at(currentContent);
  contentList.at(actual);

  QObject::disconnect(htmlw, SIGNAL(documentChanged()), this, SLOT(slotDocumentChanged())); 
  
  tmp=currentContent-actual;
  if ( tmp==1 || (tmp+contentList.count())==1)
     hc->showHtml();
  else
     hc->showHtml(true);
  QObject::connect(htmlw, SIGNAL(documentChanged() ), SLOT( slotDocumentChanged()));
  slotDocumentChanged();
}

void IrcHtml::slotForward()
{
  int actual = contentList.at();
  if (actual==currentContent)
     return;
  currentContent++;
  if (currentContent>=(int)contentList.count())
     currentContent=0;

  HtmlContent* hc=contentList.at(currentContent);
  contentList.at(actual);

  QObject::disconnect(htmlw, SIGNAL(documentChanged()), this, SLOT(slotDocumentChanged()));
  hc->showHtml(true);
  QObject::connect(htmlw, SIGNAL(documentChanged() ), SLOT( slotDocumentChanged()));
  slotDocumentChanged();
}

KHTMLWidget* IrcHtml::htmlWidget()
{
  return (KHTMLWidget*)htmlw;
}

void IrcHtml::resizeEvent(QResizeEvent*)
{
#ifdef EDEBUG
  cout << "IrcHtml::resizeEvent"<<endl;
#endif
  
  int ver=0, hor=0;
  if (sbHor!=0L){
     ver=16;
     hor=16;
  }
  window->setGeometry(0,
		      0,
		      width(),
		      height());
  sbVer->setGeometry (width()- 16, 
		      0, 
		      16,
		      height()-hor);
  if (sbHor!=0L)
     sbHor->setGeometry(0, 
			height()-16,
			width()-ver,
			16);
  
  htmlw->setGeometry(1,
		     1,
		     width() -16-1,
		     height()-ver-1);
}

#include "ircHtml.moc"




