package Crypt::Bcrypt;
$Crypt::Bcrypt::VERSION = '0.009';
use strict;
use warnings;

use XSLoader;
XSLoader::load('Crypt::Bcrypt');

use Exporter 5.57 'import';
our @EXPORT_OK = qw(bcrypt bcrypt_check bcrypt_hashed bcrypt_check_hashed bcrypt_needs_rehash);

use Digest::SHA 'hmac_sha256';
use MIME::Base64 2.21 qw(encode_base64);

sub bcrypt {
	my ($password, $subtype, $cost, $salt) = @_;
	die "Unknown subtype $subtype" if $subtype !~ /^2[abxy]$/;
	die "Invalid cost factor $cost" if $cost < 4 || $cost > 31;
	die "Salt must be 16 bytes" if length $salt != 16;
	my $encoded_salt = encode_base64($salt, "");
	$encoded_salt =~ tr{A-Za-z0-9+/=}{./A-Za-z0-9}d;
	return _bcrypt_hashpw($password, sprintf '$%s$%02d$%s', $subtype, $cost, $encoded_salt);
}

my $subtype_qr = qr/2[abxy]/;
my $cost_qr = qr/\d{2}/;
my $salt_qr = qr{ [./A-Za-z0-9]{22} }x;

sub bcrypt_hashed {
	my ($password, $subtype, $cost, $salt, $hash_algorithm) = @_;
	if ($hash_algorithm) {
		(my $encoded_salt = encode_base64($salt, "")) =~ tr{A-Za-z0-9+/=}{./A-Za-z0-9}d;
		my $hashed_password = encode_base64(hmac_sha256($password, $encoded_salt), "");
		my $hash = bcrypt($hashed_password, $subtype, $cost, $salt);
		$hash =~ s{ ^ \$ ($subtype_qr) \$ ($cost_qr) \$ ($salt_qr) }{\$bcrypt-sha256\$v=2,t=$1,r=$2\$$3\$}x or die $hash;
		return $hash;
	}
	else {
		bcrypt($password, $subtype, $cost, $salt);
	}
}

sub bcrypt_check_hashed {
	my ($password, $hash) = @_;
	if ($hash =~ s/ ^ \$ bcrypt-sha256 \$ v=2,t=($subtype_qr),r=($cost_qr) \$ ($salt_qr) \$ /\$$1\$$2\$$3/x) {
		return bcrypt_check(encode_base64(hmac_sha256($password, $3), ""), $hash);
	}
	else {
		bcrypt_check($password, $hash);
	}
}

sub _get_parameters {
	my ($hash) = @_;
	if ($hash =~ / \A \$ ($subtype_qr) \$ ($cost_qr) \$ /x) {
		return ($1, $2, '');
	}
	elsif ($hash =~ / ^ \$ bcrypt-sha256 \$ v=2,t=($subtype_qr),r=($cost_qr) \$ /x) {
		return ($1, $2, 'sha256');
	}
	return ('', 0, '');
}

sub bcrypt_needs_rehash {
	my ($hash, $wanted_subtype, $wanted_cost, $wanted_hash) = @_;
	my ($my_subtype, $my_cost, $my_hash) = _get_parameters($hash);
	return $my_subtype ne $wanted_subtype || $my_cost != $wanted_cost || $my_hash ne ($wanted_hash || '');
}

1;

# ABSTRACT: A modern bcrypt implementation

__END__

=pod

=encoding UTF-8

=head1 NAME

Crypt::Bcrypt - A modern bcrypt implementation

=head1 VERSION

version 0.009

=head1 SYNOPSIS

 use Crypt::Bcrypt qw/bcrypt bcrypt_check/;

 my $hash = bcrypt($password, '2b', 12, $salt);

 if (bcrypt_check($password, $hash)) {
    ...
 }

=head1 DESCRIPTION

This module provides a modern and user-friendly implementation of the bcrypt password hash.

Note that in bcrypt passwords may only contain 72 characters. It may seem tempting to prehash the password before bcrypting it but that may make it vulnerable to password shucking, a salted solution (for example using a MAC) should be used instead if one wants to support large passwords.

The password is always expected to come as a (utf8-encoded) byte-string.

=head1 FUNCTIONS

=head2 bcrypt($password, $subtype, $cost, $salt)

This computes the bcrypt hash for C<$password> in C<$subtype>, with C<$cost> and C<$salt>.

Valid subtypes are:

=over 4

=item * C<2b>

This is the subtype the rest of the world has been using since 2014, you should use this unless you have a very specific reason to use something else.

=item * C<2a>

This is an old and subtly buggy version of bcrypt. This is mainly useful for Crypt::Eksblowfish compatibility.

=item * C<2y>

This type is considered equivalent to C<2b>, and is only commonly used on php.

=item * C<2x>

This is a very broken version that is only useful for compatibility with ancient php versions.

=back

C<$cost> must be between 4 and 31 (inclusive). C<$salt> must be exactly 16 bytes.

=head2 bcrypt_check($password, $hash)

This checks if the C<$password> satisfies the C<$hash>, and does so in a timing-safe manner.

=head2 bcrypt_hashed($password, $subtype, $cost, $salt, $hash_algorithm)

This works like the C<bcrypt> functions, but pre-hashes the password using the specified hash. This is mainly useful to get around the 72 character limit. Currently only sha256 is supported. This uses a salt-keyed hash to prevent password shucking. If C<$hash_algorithm> is an empty string it will perform a normal C<bcrypt> operation.

=head2 bcrypt_check_hashed($password, $hash)

This verifies pre-hashed passwords as generated by C<bcrypt_hashed>.

=head2 bcrypt_needs_rehash($hash, $wanted_subtype, $wanted_cost, $wanted_hash = '')

This returns true if the bcrypt hash uses a different subtype, cost or hash algorithm than desired.

=head1 SEE OTHER

=over 4

=item * L<Crypt::Passphrase|Crypt::Passphrase>

This is usually a better approach to managing your passwords, it can use this module via L<Crypt::Passphrase::Bcrypt|Crypt::Passphrase::Bcrypt>. It facilitates upgrading the algorithm parameters or even the algorithm itself.

=item * L<Crypt::Eksblowfish::Bcrypt|Crypt::Eksblowfish::Bcrypt>

This also offers bcrypt, but only supports the C<2a> subtype.

=back

=head1 AUTHOR

Leon Timmermans <leont@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021 by Leon Timmermans.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
